/**************************************************************************************

Copyright (c) Hilscher Gesellschaft fuer Systemautomation mbH. All Rights Reserved.

***************************************************************************************

  $Id: OS_Spi.c 12210 2018-07-24 08:36:50Z Robert $:

  Description:
    Operating system depending implementation of SPI functions

  Changes:
    Date        Description
    -----------------------------------------------------------------------------------
    2018-07-24  Fixed PC-Lint warnings
    2018-07-18  Moved functions from general ToolkitSample.c into own module

**************************************************************************************/

#include "cifXErrors.h"

/* rcX includes */
#include <rX_Includes.h>
#include "ToolkitSample.h"


/*****************************************************************************/
/*! Initialize SPI components
*   \param pvOSDependent OS Dependent parameter to identify card
*   \return CIFX_NO_ERROR on success                                         */
/*****************************************************************************/
long OS_SpiInit(void* pvOSDependent)
{
  RCX_DEVICEINSTANCE_T* ptrcXInst = (RCX_DEVICEINSTANCE_T*) pvOSDependent;
  long ulRet = CIFX_NO_ERROR;

  if ( (RX_OK != Drv_SpiIdentifySpi("SERDPM", 0, &ptrcXInst->hSerDPM)) ||
       (RX_OK != Drv_SpiInitializeSpi(ptrcXInst->hSerDPM))             ||
       (NULL  == (ptrcXInst->hSerLock = OS_CreateLock()))              )
  {
    ulRet = CIFX_FUNCTION_FAILED;
  }

  return ulRet;
}

/*****************************************************************************/
/*! Assert chip select
*   \param pvOSDependent OS Dependent parameter to identify card             */
/*****************************************************************************/
void OS_SpiAssert(void* pvOSDependent)
{
  RCX_DEVICEINSTANCE_T* ptrcXInst = (RCX_DEVICEINSTANCE_T*)pvOSDependent;
  (void)Drv_SpiOpen(ptrcXInst->hSerDPM, RX_INFINITE);
  (void)Drv_SpiSlaveSelect(ptrcXInst->hSerDPM);
}

/*****************************************************************************/
/*! Deassert chip select
*   \param pvOSDependent OS Dependent parameter to identify card             */
/*****************************************************************************/
void OS_SpiDeassert(void* pvOSDependent)
{
  RCX_DEVICEINSTANCE_T* ptrcXInst = (RCX_DEVICEINSTANCE_T*)pvOSDependent;
  (void)Drv_SpiSlaveDeselect(ptrcXInst->hSerDPM);
  (void)Drv_SpiClose(ptrcXInst->hSerDPM);
}

/*****************************************************************************/
/*! Lock the SPI bus
*   \param pvOSDependent OS Dependent parameter to identify card             */
/*****************************************************************************/
void OS_SpiLock(void* pvOSDependent)
{
  RCX_DEVICEINSTANCE_T* ptrcXInst = (RCX_DEVICEINSTANCE_T*) pvOSDependent;
  (void)OS_EnterLock(ptrcXInst->hSerLock);
}

/*****************************************************************************/
/*! Unlock the SPI bus
*   \param pvOSDependent OS Dependent parameter to identify card             */
/*****************************************************************************/
void OS_SpiUnlock(void* pvOSDependent)
{
  RCX_DEVICEINSTANCE_T* ptrcXInst = (RCX_DEVICEINSTANCE_T*) pvOSDependent;
  (void)OS_LeaveLock(ptrcXInst->hSerLock);
}

/*****************************************************************************/
/*! Transfer byte stream via SPI
*   \param pvOSDependent OS Dependent parameter to identify card
*   \param pbSend        Send buffer (NULL for polling)
*   \param pbRecv        Receive buffer (NULL if discard)
*   \param ulLen         Length of SPI transfer                              */
/*****************************************************************************/
void OS_SpiTransfer(void* pvOSDependent, uint8_t* pbSend, uint8_t* pbRecv, uint32_t ulLen)
{
  RCX_DEVICEINSTANCE_T* ptrcXInst = (RCX_DEVICEINSTANCE_T*)pvOSDependent;
  (void)Drv_SpiSendReceive(ptrcXInst->hSerDPM, ulLen, pbSend, pbRecv);
}
